<?php

namespace App\Http\Conversations;

use App\Models\TherapistCommunity;
use App\Models\TherapistCommunityCategory;
use App\Models\Therapist;
use App\Models\TherapistCategory;
use App\Models\Service;
use App\Models\ServiceCategory;
use BotMan\BotMan\Messages\Incoming\Answer;
use BotMan\BotMan\Messages\Outgoing\Question;
use BotMan\BotMan\Messages\Outgoing\Actions\Button;
use BotMan\BotMan\Messages\Conversations\Conversation;
use BotMan\BotMan\Messages\Outgoing\OutgoingMessage;
use Exception;
use Illuminate\Support\Facades\Log;

class ServicesConversations extends Conversation
{
    /**
     * Start the conversation
     *
     * @return mixed
     */

    protected $name = '';
    protected $email = '';
    protected $service = '';
    protected $list_of_keywords = [];

    public function __construct()
    {
        // Load keywords from the external file
        $this->list_of_keywords = require __DIR__ . '/ListOfKeywords.php';
    }

    public function run()
    {
        $this->ask('Hi there! What is your Name?', function (Answer $answer) {
            $this->name = $answer->getText();
            if (empty($this->name)) {
                $this->say('Name cannot be empty.');
                return $this->repeat();
            }
            $this->say('Hi! Nice to meet you, ' . $this->name);
            $this->askEmail();
        });
    }

    public function askEmail()
    {
        $this->ask('What is your Email?', function (Answer $answer) {
            $this->email = $answer->getText();
            if (!filter_var($this->email, FILTER_VALIDATE_EMAIL)) {
                $this->say('Please enter a valid email address.');
                return $this->repeat();
            }
            $this->askForOptions();
        });
    }

    protected function askForOptions()
    {

        $questions = Question::create('Please Select your needy Option?')
            ->fallback('Unable to ask question')->addButtons([
                Button::create('I need a Therapist')->value('therapist'),
                Button::create('I need a Therapyfirm')->value('therapistcommunity'),
                Button::create('I need a Service')->value('service'),
            ]);
        $this->ask($questions, function (Answer $answer) {
            if (!$answer->isInteractiveMessageReply()) {
                $this->say('Please select an option.');
                return $this->repeat();
            }
            $selected_option = $answer->getValue();
            $this->say('You selected : ' . $selected_option);
            if ($selected_option == 'therapist') {
                $this->askTherapistCategories();
            } elseif ($selected_option == 'therapistcommunity') {
                $this->askTherapyfirmCategories();
            } else {
                $this->askServiceCategories();
            }
        });
    }

    protected function askTherapistCategories()
    {
        try {
            $therapist_categories = TherapistCategory::active()->get();
            if ($therapist_categories->isEmpty()) {
                $this->say('No therapist categories available at the moment.');
                return;
            }

            $questions = Question::create('Which type of therapist do you want?')
                ->fallback('Unable to ask question');

            foreach ($therapist_categories as $therapist_category) {
                $questions->addButton(Button::create($therapist_category->name)->value($therapist_category->id));
            }

            $this->ask($questions, function (Answer $answer) {
                if (!$answer->isInteractiveMessageReply()) {
                    $this->say('Please select a therapist category.');
                    return $this->repeat();
                }

                $selectedCategoryId = $answer->getValue();
                $this->say('You selected category ID: ' . $selectedCategoryId);
                $this->therapistSuggestions($selectedCategoryId, 'category');
            });
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving service categories. Please try again later.');
            Log::error('Error fetching service categories: ' . $e->getMessage());
        }
    }


    protected function askTherapyfirmCategories()
    {
        try {
            $therapistcommunity_categories = TherapistCommunityCategory::active()->get();
            if ($therapistcommunity_categories->isEmpty()) {
                $this->say('No therapistcommunity categories available at the moment.');
                return;
            }

            $questions = Question::create('Which type of therapistcommunity do you want?')
                ->fallback('Unable to ask question');

            foreach ($therapistcommunity_categories as $therapistcommunity_category) {
                $questions->addButton(Button::create($therapistcommunity_category->name)->value($therapistcommunity_category->id));
            }

            $this->ask($questions, function (Answer $answer) {
                if (!$answer->isInteractiveMessageReply()) {
                    $this->say('Please select a therapistcommunity category.');
                    return $this->repeat();
                }

                $selectedCategoryId = $answer->getValue();
                $this->say('You selected category ID: ' . $selectedCategoryId);
                $this->therapistcommunitySuggestions($selectedCategoryId);

            });
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving service categories. Please try again later.');
            Log::error('Error fetching service categories: ' . $e->getMessage());
        }
    }





    protected function askServiceCategories()
    {
        try {
            $service_categories = ServiceCategory::all();
            if ($service_categories->isEmpty()) {
                $this->say('No service categories available at the moment.');
                return;
            }

            $questions = Question::create('Which type of service do you want?')
                ->fallback('Unable to ask question');

            foreach ($service_categories as $service_category) {
                $questions->addButton(Button::create($service_category->name)->value($service_category->id));
            }

            $this->ask($questions, function (Answer $answer) {
                if (!$answer->isInteractiveMessageReply()) {
                    $this->say('Please select a service category.');
                    return $this->repeat();
                }

                $selectedCategoryId = $answer->getValue();
                $this->say('You selected category ID: ' . $selectedCategoryId);
                $this->askService($selectedCategoryId);
            });
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving service categories. Please try again later.');
            Log::error('Error fetching service categories: ' . $e->getMessage());
        }
    }

    protected function askService($selectedCategoryId)
    {
        try {
            $services = Service::where('service_category_id', $selectedCategoryId)->get();
            if ($services->isEmpty()) {
                $this->say('No services available for this category.');
                return;
            }

            $questions = Question::create('List of Services');

            foreach ($services as $service) {
                $questions->addButton(Button::create($service->name)->value($service->id));
            }

            $this->ask($questions, function (Answer $answer) {
                if (!$answer->isInteractiveMessageReply()) {
                    $this->say('Please select a service.');
                    return $this->repeat();
                }

                $service_id = $answer->getValue();
                $this->say('You selected Service ID: ' . $service_id);
                $this->therapistSuggestions($service_id, 'service');
            });
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving services. Please try again later.');
            Log::error('Error fetching services: ' . $e->getMessage());
        }
    }

    protected function therapistSuggestions($id, $type)
    {
        try {
            $therapists = Therapist::limit(5);
            if ($type == 'category') {
                $therapists->whereHas('therapist_categories', function ($query) use ($id) {
                    $query->where('therapist_category_id', $id);
                });
            } else {
                $therapists = $therapists->has('therapist_services');
            }
            $therapists = $therapists->get();

            if ($therapists->isEmpty()) {
                $this->say('No therapists found for this service.');
                return;
            }


            $listItems = '';
            $this->say('list of Therapists: ');
            foreach ($therapists as $therapist) {
                $listItems .= '<li>Name: <a href="' . url('/therapist/profile/' . $therapist->user_name) . '" target="_blank">' . $therapist->name . '</a></li>';
            }
            $messageContent = '<ul>' . $listItems . '</ul>';
            $message = OutgoingMessage::create($messageContent);
            $this->say($message);
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving therapists. Please try again later.');
            Log::error('Error fetching therapists: ' . $e->getMessage());
        }
    }

    protected function therapistcommunitySuggestions($id)
    {
        try {
            $therapistcommunitys = TherapistCommunity::limit(5);
            $therapistcommunitys->whereHas('therapist_community_categories', function ($query) use ($id) {
                $query->where('therapist_community_id', $id);
            });

            $therapistcommunitys = $therapistcommunitys->get();

            if ($therapistcommunitys->isEmpty()) {
                $this->say('No therapistcommunitys found for this service.');
                return;
            }

            $listItems = '';
            $this->say('list of therapyforms: ');
            foreach ($therapistcommunitys as $therapistcommunity) {
                $listItems .= '<li>Name: <a href="' . url('/therapist_community/profile/' . $therapistcommunity->user_name) . '" target="_blank">' . $therapistcommunity->name . '</a></li>';
            }
            $messageContent = '<ul>' . $listItems . '</ul>';
            $message = OutgoingMessage::create($messageContent);
            $this->say($message);
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving therapistcommunitys. Please try again later.');
            Log::error('Error fetching therapistcommunitys: ' . $e->getMessage());
        }
    }
}
